#include <pic.h>

/* Basic A2D sample code for an PIC16F87x device.
 * This code willl set up the A2D module to return an
 * 8-Bit result. If greater precision is needed, a 10-Bit
 * result can be returned if read_a2d() is modified to
 * return the short ((ADRESH<<8)+(ADRESL)). Note also that
 * ADCON1 should be set to 0x80 in the init_a2d() routine
 * instead of zero.
 *
 * This code will sample an A2D value on analog port RA2, and it's value
 * will be used to drive a 2-dig 7seg display on PORTC.
 *
 * This project can be demonstrated on the Microchip PICDEM2 board.
 */

#define SEL_NONE        (PORTB = 0x60);
#define SEL_DIGIT1		(PORTB = 0x40);
#define SEL_DIGIT2		(PORTB = 0x20);

// __CONFIG(DEBUGEN & WDTDIS & LVPDIS);	// Setup the configuration word for ise with ICD2

/* Sample code to set up the A2D module */
// ADFM = 0, left, = 1, right
void init_a2d(void){
	ADCON0=0x80;	// select left justify result. A/D port configuration 0
	ADCON1=0;       // select Fosc/2
	ADON=1;			// turn on the A2D conversion module
}

unsigned char dot=0;
unsigned char display=1;

void delay(void);
void delaySH(void);
unsigned int voltx( unsigned int val );

void disp( unsigned int val, unsigned char digit );

// ADFM = 0, left, = 1, right
// VCFG = 0, VDD,  = 1, VREF
// ADCON
// 10 bitter

/* Return a right-justified 10 bit result */
unsigned int read_a2d(unsigned char channel)
{
	unsigned int adc;

	channel &= 0x07;			// truncate channel to 3 bits
	ADCON0  &= 0x83;			// clear current channel select
	ADCON0  |= (channel<<2);	// apply the new channel select
	GODONE  =  1;				// initiate conversion on the selected channel
	while(GODONE)continue;
	
    // adc = ADRESH;

    adc = ((ADRESH<<8)+(ADRESL));		// return 10 MSB of the result

    return adc;
}

/* Entry point */
void main(void)
{
	unsigned int i;
	unsigned char x;
    unsigned int val=0;
    unsigned int meas=0;

    unsigned int volt;
    unsigned int curr;
	unsigned int last_meas = 0;
	unsigned int diff;
    unsigned long diff_ave;
	unsigned int diff_cnt=0;

    unsigned char bpress = 0;
	unsigned char last_bpress = 0;

	unsigned long display_cnt_upw=0;

	init_a2d();	// initialize the A2D module
	GIE=0;		// we don't want interrupts

	// RB7 is an input
	TRISB=0x80;

	// RB5 is 1_CATH
	// RB6 is 2_CATH
	// RB% is also AN11, so ANS11 of ANSELH must be set to 0, defaults to 1
	ANSELH = 0;

	// RC7 is output and is driving LEDs
 
    TRISC = 0x00;
    PORTC = 0x00;
	
	for(;;)
    {
		volt=read_a2d(2);		    // sample the analog value on RA2 (voltage)
		curr=read_a2d(3);		    // sample the analog value on RA3 (voltage)
		diff=curr-volt;

		diff_ave += diff;
		diff_cnt += 1;

		if (diff_cnt==16)
	    {
			diff_ave >>= 4;
		}

        // Check push button 0 = pushed
		bpress=RB7;
		if (!bpress)
		{
#if 1
		  diff = diff_ave;
#endif
          display = 1;
		  display_cnt_upw = 0;

		  dot=1;
          disp(volt>>8,1);
          delay();
          disp(curr>>8,2);
          delay();

          disp((volt>>4)&0xf,1);
          delay();
          disp((curr>>4)&0xf,2);
          delay();

          disp(volt&0xf,1);
          delay();
          disp(curr&0xf,2);
		  delay();
        }
        else
        {
		  meas = voltx(volt);

		  dot=1;
          disp(meas>>4,1);
          delaySH();
		  disp(meas&0xf,2);
		  delaySH();

			SEL_NONE;
		  delaySH();
        }

#if 0
		// Count down and turn off display
		if (display)
        {
            if (display_cnt_upw<200)
                display_cnt_upw++;

			else
				display=0;
		}

		// If voltage changes or button changes
		if ( (meas != last_meas) || (bpress != last_bpress) )
		{
			display = 1;
			display_cnt_upw = 0;
		}

		// Save Last
		last_meas   = meas;
		last_bpress = bpress;
#endif
	}
}

const unsigned int ddd[] = 
{
	0x3f,		// 0
	0x06,		// 1
	0x5b,		// 2
	0x4f,		// 3
	0x66,		// 4
	0x6d,		// 5
	0x7d,		// 6
	0x07,		// 7
	0x7f,		// 8
	0x6f,		// 9

	0x77,		// A
	0x7c,		// B
	0x39,		// C
	0x5e,		// D
	0x79,		// E
	0x71,		// F
};

void disp( unsigned int val, unsigned char digit )
{
	unsigned char pc;

	SEL_NONE;
	if (!display)
	{
		SEL_NONE;
	}

    if (val>0xf)
        val=0xf;

	pc = ddd[val];

	if (dot)
        pc |= 0x80;

    dot = 0;

	PORTC = pc;

	if (digit==1)
	   SEL_DIGIT1;
	if (digit==2)
       SEL_DIGIT2;
}

typedef struct _bob {

	unsigned int meas;
	unsigned int volt;
} MC;

const MC meas_convert[] = 
{
    { 0x100, 0x00 },
    { 0x159, 0x17 },
    { 0x16D, 0x18 },
    { 0x181, 0x19 },
    { 0x195, 0x20 },
    { 0x1AA, 0x21 },
    { 0x1BE, 0x22 },
    { 0x1D3, 0x23 },
    { 0x1E7, 0x24 },
    { 0x1FC, 0x25 },
    { 0x211, 0x26 },
    { 0x225, 0x27 },
    { 0x23A, 0x28 },
    { 0x24E, 0x29 },
    { 0x263, 0x30 },
    { 0x277, 0x31 },
    { 0x28C, 0x32 },
    { 0x2A0, 0x33 },
    { 0x2B5, 0x34 },
    { 0x2CA, 0x35 },
    { 0x2DE, 0x36 },
    { 0x2F3, 0x37 },
    { 0x307, 0x38 },
    { 0x31C, 0x39 },
    { 0x331, 0x40 },
    { 0x345, 0x41 },
    { 0x35A, 0x42 },
    { 0x36E, 0x43 },
    { 0x383, 0x44 },
    { 0x397, 0x45 },
    { 0x3FF, 0x46 },
};

unsigned int voltx( unsigned int val )
{
	unsigned int iii;
	unsigned int meas;

    for (iii=0; iii< sizeof(meas_convert)/sizeof(meas_convert[0]); iii++)
    {
        meas = meas_convert[iii].volt;
        if( val < meas_convert[iii].meas )
            break;
    }

    return meas;
}

void delay(void)
{
    volatile unsigned long i;
	for(i=0; i<20000; i++);
}

void delaySH(void)
{
    volatile unsigned long i;
	for(i=0; i<100; i++);
}


// #define 0x80 CROSS